<?php
// includes/functions.php
require_once __DIR__ . '/db.php';

function getServerStats($server_port = 27015) {
    $pdo = getDB();
    // Get latest log for specific server
    $stmt = $pdo->prepare("SELECT * FROM poll_logs WHERE server_port = ? ORDER BY id DESC LIMIT 1");
    $stmt->execute([$server_port]);
    return $stmt->fetch();
}

function getTopPlayers($limit = 10, $server_port = 27015) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM players WHERE server_port = ? ORDER BY total_minutes DESC LIMIT ?");
    $stmt->bindValue(1, $server_port, PDO::PARAM_INT);
    $stmt->bindValue(2, $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function getPlayer($steam_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM players WHERE steam_id = ?");
    $stmt->execute([$steam_id]);
    return $stmt->fetch();
}

function getPlayerByNick($nick) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM players WHERE last_nick = ? OR steam_id = ?");
    $stmt->execute([$nick, "NAME_" . $nick]);
    return $stmt->fetch();
}

function getPlayerDailyStats($steam_id, $days = 7) {
    $pdo = getDB();
    
    // Safety cap at 30 days
    $days = min(30, max(1, $days));
    
    $stats = [];
    for ($i = $days; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $stats[$date] = ['date' => $date, 'minutes_played' => 0, 'kills_count' => 0];
    }

    try {
        $stmt = $pdo->prepare("SELECT * FROM playtime_daily WHERE steam_id = ? AND date >= ? ORDER BY date ASC");
        $stmt->execute([$steam_id, date('Y-m-d', strtotime("-$days days"))]);
        $rows = $stmt->fetchAll();

        foreach ($rows as $row) {
            if (isset($stats[$row['date']])) {
                $stats[$row['date']] = $row;
            }
        }
    } catch (Exception $e) {
        // Fallback or log
    }

    return array_values($stats);
}

function getPlayerDailyStatsByNick($nick, $days = 7) {
    $player = getPlayerByNick($nick);
    if (!$player) return [];
    return getPlayerDailyStats($player['steam_id'], $days);
}

function searchPlayers($query) {
    $pdo = getDB();
    $term = "%$query%";
    // Allow searching for single character nicknames
    $stmt = $pdo->prepare("SELECT * FROM players WHERE last_nick LIKE ? OR steam_id LIKE ? ORDER BY total_minutes DESC LIMIT 50");
    $stmt->execute([$term, $term]);
    return $stmt->fetchAll();
}

function formatMinutes($min) {
    $hours = floor($min / 60);
    $minutes = $min % 60;
    return "{$hours}h {$minutes}m";
}

function getOnlinePlayers($server_port = 27015) {
    $pdo = getDB();
    // Use 60 second threshold for stable real-time active player detection
    $threshold = date('Y-m-d H:i:s', strtotime('-60 seconds'));
    $now = date('Y-m-d H:i:s');
    
    // Only show players seen in the last 60 seconds for specific server
    $stmt = $pdo->prepare("SELECT * FROM players WHERE server_port = ? AND last_seen > ? AND last_seen <= ? ORDER BY last_nick ASC");
    $stmt->execute([$server_port, $threshold, $now]);
    return $stmt->fetchAll();
}

function getAllPlayers($limit = 50, $offset = 0, $server_port = null) {
    $pdo = getDB();
    if ($server_port !== null) {
        $stmt = $pdo->prepare("SELECT * FROM players WHERE server_port = ? ORDER BY total_minutes DESC LIMIT ? OFFSET ?");
        $stmt->bindValue(1, $server_port, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->bindValue(3, $offset, PDO::PARAM_INT);
    } else {
        $stmt = $pdo->prepare("SELECT * FROM players ORDER BY total_minutes DESC LIMIT ? OFFSET ?");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->bindValue(2, $offset, PDO::PARAM_INT);
    }
    $stmt->execute();
    return $stmt->fetchAll();
}

function getTotalPlayersCount($server_port = null) {
    $pdo = getDB();
    if ($server_port !== null) {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM players WHERE server_port = ?");
        $stmt->execute([$server_port]);
        return $stmt->fetchColumn();
    }
    return $pdo->query("SELECT COUNT(*) FROM players")->fetchColumn();
}

function getPlayerRank($steam_id) {
    $pdo = getDB();
    // Count how many players have more minutes than this player
    // +1 because if 0 people have more, you are rank 1.
    $stmt = $pdo->prepare("SELECT COUNT(*) + 1 FROM players WHERE total_minutes > (SELECT total_minutes FROM players WHERE steam_id = ?)");
    $stmt->execute([$steam_id]);
    return $stmt->fetchColumn();
}

function getAdminLevel($nick, $server_port = null) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT level, server_access FROM tracked_admins WHERE nickname = ?");
    $stmt->execute([$nick]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$row) return false;
    
    // If context is provided, verify access
    if ($server_port !== null) {
        $contextName = 'public';
        if ($server_port == 27016) $contextName = 'afk';
        if ($server_port == 27018) $contextName = 'deathmatch';
        
        $access = explode(',', $row['server_access']);
        // Upper Management has access to EVERYTHING
        if ($row['level'] !== 'Upper Management' && $row['server_access'] !== 'all' && !in_array($contextName, $access)) {
            return false; // Not an admin in this context
        }
    }
    
    return $row['level']; 
}

function getPlaytimeForRange($steam_id, $startDate, $endDate, $server_port = null) {
    $pdo = getDB();
    if ($server_port !== null) {
        $stmt = $pdo->prepare("SELECT SUM(minutes_played) FROM playtime_daily WHERE steam_id = ? AND date >= ? AND date <= ? AND server_port = ?");
        $stmt->execute([$steam_id, $startDate, $endDate, $server_port]);
    } else {
        $stmt = $pdo->prepare("SELECT SUM(minutes_played) FROM playtime_daily WHERE steam_id = ? AND date >= ? AND date <= ?");
        $stmt->execute([$steam_id, $startDate, $endDate]);
    }
    return (int)$stmt->fetchColumn();
}

function getAdminActionCountsForRange($pdo, $startDate, $endDate, $server_port = null) {
    $start = (strlen($startDate) <= 10) ? "$startDate 00:00:00" : $startDate;
    $end = (strlen($endDate) <= 10) ? "$endDate 23:59:59" : $endDate;
    
    if ($server_port !== null) {
        $stmt = $pdo->prepare("SELECT admin_nick, action_type, COUNT(*) as cnt FROM admin_actions WHERE created_at >= ? AND created_at <= ? AND server_port = ? GROUP BY admin_nick, action_type");
        $stmt->execute([$start, $end, $server_port]);
    } else {
        $stmt = $pdo->prepare("SELECT admin_nick, action_type, COUNT(*) as cnt FROM admin_actions WHERE created_at >= ? AND created_at <= ? GROUP BY admin_nick, action_type");
        $stmt->execute([$start, $end]);
    }
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getPlayerDailyStatsForRange($steam_id, $startDate, $endDate) {
    $pdo = getDB();
    $stats = [];
    
    $start = new DateTime($startDate);
    $end = new DateTime($endDate);
    $interval = new DateInterval('P1D');
    $period = new DatePeriod($start, $interval, $end->modify('+1 day'));

    foreach ($period as $date) {
        $d = $date->format('Y-m-d');
        $stats[$d] = ['date' => $d, 'minutes_played' => 0, 'kills_count' => 0];
    }

    try {
        $stmt = $pdo->prepare("SELECT * FROM playtime_daily WHERE steam_id = ? AND date >= ? AND date <= ? ORDER BY date ASC");
        $stmt->execute([$steam_id, $startDate, $endDate]);
        $rows = $stmt->fetchAll();

        foreach ($rows as $row) {
            if (isset($stats[$row['date']])) {
                $stats[$row['date']] = $row;
            }
        }
    } catch (Exception $e) {}

    return array_values($stats);
}

function syncPlayerTotalPlaytime($steam_id) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT SUM(minutes_played) FROM playtime_daily WHERE steam_id = ?");
    $stmt->execute([$steam_id]);
    $sum = (int)$stmt->fetchColumn();
    
    // Check if player exists
    $stmtCheck = $pdo->prepare("SELECT id FROM players WHERE steam_id = ?");
    $stmtCheck->execute([$steam_id]);
    if ($stmtCheck->fetch()) {
        $pdo->prepare("UPDATE players SET total_minutes = ? WHERE steam_id = ?")->execute([$sum, $steam_id]);
    } else {
        // Create placeholder if nickname is derivable
        $nick = str_replace("NAME_", "", $steam_id);
        $now = date('Y-m-d H:i:s');
        $pdo->prepare("INSERT INTO players (steam_id, last_nick, total_minutes, first_seen, last_seen) VALUES (?, ?, ?, ?, ?)")
            ->execute([$steam_id, $nick, $sum, $now, $now]);
    }
    return $sum;
}

function getPlayerDailyMinutesMap($steam_id, $days = 7) {
    $pdo = getDB();
    $stats = [];
    $startDate = date('Y-m-d', strtotime("-$days days"));
    $endDate = date('Y-m-d');
    
    $stmt = $pdo->prepare("SELECT date, minutes_played FROM playtime_daily WHERE steam_id = ? AND date >= ? AND date <= ?");
    $stmt->execute([$steam_id, $startDate, $endDate]);
    $rows = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Fill gaps
    $map = [];
    for ($i = $days; $i >= 0; $i--) {
        $d = date('Y-m-d', strtotime("-$i days"));
        $map[$d] = (int)($rows[$d] ?? 0);
    }
    return $map;
}

// Get server port from steam_id (e.g., "NAME_Player_27016" returns 27016)
function getServerPortFromSteamId($steam_id) {
    // Format: NAME_playername_27016 OR NAME_playername (old public format)
    $parts = explode('_', $steam_id);
    $lastPart = end($parts);
    // Check if last part is a valid port number
    if (is_numeric($lastPart) && $lastPart >= 27015 && $lastPart <= 27030) {
        return (int)$lastPart;
    }
    // If no port suffix, it's the old public server format
    return 27015;
}

// Get server name from port
function getServerName($port) {
    switch($port) {
        case 27015: return 'DSGC Public';
        case 27016: return 'DSGC AFK Server';
        case 27018: return 'DSGC Deathmatch';
        default: return 'DSGC Server';
    }
}

/**
 * Get player by nickname and server port (Context Aware)
 */
function getPlayerByNickAndPort($nickname, $port = 27015) {
    $pdo = getDB();
    $stmt = $pdo->prepare("SELECT * FROM players WHERE last_nick = ? AND server_port = ? ORDER BY last_seen DESC LIMIT 1");
    $stmt->execute([$nickname, $port]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get active admins for a specific server
 * This function checks both online players and server access permissions
 */
function getActiveAdminsForServer($server_port = 27015) {
    $pdo = getDB();
    
    // Determine server context
    $contextName = 'public';
    if ($server_port == 27016) $contextName = 'afk';
    if ($server_port == 27018) $contextName = 'deathmatch';
    
    // Fetch currently active players (seen in last 60 seconds)
    $threshold_active = date('Y-m-d H:i:s', strtotime('-60 seconds'));
    $stmt_players = $pdo->prepare("SELECT last_nick FROM players WHERE server_port = ? AND last_seen > ?");
    $stmt_players->execute([$server_port, $threshold_active]);
    $onlinePlayers = $stmt_players->fetchAll(PDO::FETCH_COLUMN);
    
    // Early return if no players online
    if (empty($onlinePlayers)) {
        return [];
    }
    
    // Fetch all tracked admins
    $stmt_admins = $pdo->query("SELECT nickname, level, server_access FROM tracked_admins");
    $trackedAdmins = $stmt_admins->fetchAll(PDO::FETCH_ASSOC);
    
    // Find active admins respecting server access
    $activeAdmins = [];
    foreach ($onlinePlayers as $playerNick) {
        foreach ($trackedAdmins as $admin) {
            // Case-insensitive nickname match
            if (strcasecmp(trim($playerNick), trim($admin['nickname'])) === 0) {
                // Check if admin has access to this server
                $access = explode(',', $admin['server_access']);
                
                // Upper Management has access to everything
                // OR server_access is 'all'
                // OR the specific context is in the access list
                if ($admin['level'] === 'Upper Management' || 
                    $admin['server_access'] === 'all' || 
                    in_array($contextName, $access)) {
                    $activeAdmins[] = [
                        'nick' => $playerNick,
                        'level' => $admin['level']
                    ];
                }
                break; // Found match, no need to check other admins
            }
        }
    }
    
    return $activeAdmins;
}
?>
